<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AWC Children Monthly Data</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
        }

        .header {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            margin-bottom: 30px;
        }

        .header h1 {
            color: #333;
            font-size: 32px;
            margin-bottom: 10px;
        }

        .header p {
            color: #666;
            font-size: 16px;
        }

        .form-card {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            margin-bottom: 30px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
            font-size: 14px;
        }

        .form-group select {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 6px;
            font-size: 15px;
            color: #333;
            background: white;
            transition: all 0.3s ease;
            cursor: pointer;
        }

        .form-group select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .form-group select:disabled {
            background: #f5f5f5;
            cursor: not-allowed;
            opacity: 0.6;
        }

        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }

        .btn {
            padding: 12px 30px;
            border: none;
            border-radius: 6px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-primary {
            background: #667eea;
            color: white;
        }

        .btn-primary:hover:not(:disabled) {
            background: #5568d3;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }

        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .data-section {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            margin-bottom: 30px;
            display: none;
        }

        .data-section.show {
            display: block;
        }

        .data-section h2 {
            color: #333;
            font-size: 24px;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 3px solid #667eea;
        }

        .table-wrapper {
            overflow-x: auto;
            margin-top: 20px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
        }

        table thead {
            background: #667eea;
            color: white;
        }

        table th,
        table td {
            padding: 12px 15px;
            text-align: left;
            border-bottom: 1px solid #e0e0e0;
            font-size: 14px;
        }

        table th {
            font-weight: 600;
            text-transform: uppercase;
            font-size: 12px;
            letter-spacing: 0.5px;
        }

        table tbody tr:hover {
            background: #f8f9ff;
        }

        table tbody tr:last-child td {
            border-bottom: none;
        }

        .no-data {
            text-align: center;
            padding: 40px;
            color: #999;
            font-size: 16px;
        }

        .loading {
            text-align: center;
            padding: 20px;
            color: #667eea;
            font-weight: 600;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 6px;
            margin-bottom: 20px;
            display: none;
        }

        .alert.show {
            display: block;
        }

        .alert-info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }

        .alert-warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            border-radius: 8px;
            color: white;
            text-align: center;
        }

        .stat-number {
            font-size: 36px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            opacity: 0.9;
        }

        @media (max-width: 768px) {
            .header h1 {
                font-size: 24px;
            }

            .form-row {
                grid-template-columns: 1fr;
            }

            table {
                font-size: 12px;
            }

            table th,
            table td {
                padding: 8px 10px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>AWC Children Monthly Data Management</h1>
            <p>Select district and anganwadi center to view children data</p>
        </div>

        <div class="form-card">
            <div class="alert alert-info" id="alertBox"></div>
            
            <form id="searchForm">
                <div class="form-row">
                    <div class="form-group">
                        <label for="district">Select District *</label>
                        <select id="district" name="district" required>
                            <option value="">-- Select District --</option>
                            <?php foreach ($districts as $district): ?>
                                <option value="<?= $district['district_id'] ?>">
                                    <?= esc($district['district_name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="anganwadi">Select Anganwadi Center *</label>
                        <select id="anganwadi" name="anganwadi" required disabled>
                            <option value="">-- Select District First --</option>
                        </select>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary" id="searchBtn" disabled>
                    Search Children Data
                </button>
            </form>
        </div>

        <div class="data-section" id="statsSection">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number" id="totalChildren">0</div>
                    <div class="stat-label">Total Children (0-6 months)</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number" id="totalOutreach">0</div>
                    <div class="stat-label">Outreach Records</div>
                </div>
            </div>
        </div>

        <div class="data-section" id="outreachSection">
            <h2>Monthly Outreach Data</h2>
            <div class="table-wrapper">
                <table id="outreachTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>AWC Center Code</th>
                            <th>Worker Name</th>
                            <th>Address</th>
                            <th>Building Type</th>
                            <th>Location Area</th>
                            <th>Month/Year</th>
                            <th>Created At</th>
                        </tr>
                    </thead>
                    <tbody id="outreachTableBody">
                    </tbody>
                </table>
            </div>
        </div>

        <div class="data-section" id="childrenSection">
            <h2>Children Data (0-6 Months)</h2>
            <div class="table-wrapper">
                <table id="childrenTable">
                    <thead>
                        <tr>
                            <th>Child ID</th>
                            <th>Name</th>
                            <th>Gender</th>
                            <th>DOB</th>
                            <th>Age (Months)</th>
                            <th>Father Name</th>
                            <th>Mother Name</th>
                            <th>Mobile No</th>
                            <th>AWC Name</th>
                            <th>Birth Weight</th>
                            <th>Current Weight</th>
                            <th>Category</th>
                        </tr>
                    </thead>
                    <tbody id="childrenTableBody">
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <script>
        const districtSelect = document.getElementById('district');
        const anganwadiSelect = document.getElementById('anganwadi');
        const searchBtn = document.getElementById('searchBtn');
        const searchForm = document.getElementById('searchForm');

        // When district is selected, load anganwadi centers
        districtSelect.addEventListener('change', async function() {
            const districtId = this.value;
            
            // Reset anganwadi dropdown
            anganwadiSelect.innerHTML = '<option value="">-- Loading... --</option>';
            anganwadiSelect.disabled = true;
            searchBtn.disabled = true;

            if (!districtId) {
                anganwadiSelect.innerHTML = '<option value="">-- Select District First --</option>';
                return;
            }

            try {
                const formData = new FormData();
                formData.append('district_id', districtId);

                const response = await fetch('<?= base_url('home/get-anganwadi') ?>', {
                    method: 'POST',
                    body: formData
                });

                const anganwadis = await response.json();

                if (anganwadis.length === 0) {
                    anganwadiSelect.innerHTML = '<option value="">-- No Anganwadi Centers Found --</option>';
                    showAlert('No anganwadi centers found for selected district', 'warning');
                } else {
                    anganwadiSelect.innerHTML = '<option value="">-- Select Anganwadi Center --</option>';
                    anganwadis.forEach(awc => {
                        const option = document.createElement('option');
                        option.value = awc.awc_center_code;
                        option.textContent = awc.name + ' (' + awc.awc_center_code + ')';
                        anganwadiSelect.appendChild(option);
                    });
                    anganwadiSelect.disabled = false;
                }
            } catch (error) {
                console.error('Error:', error);
                anganwadiSelect.innerHTML = '<option value="">-- Error Loading Data --</option>';
                showAlert('Error loading anganwadi centers', 'warning');
            }
        });

        // Enable search button when anganwadi is selected
        anganwadiSelect.addEventListener('change', function() {
            searchBtn.disabled = !this.value;
        });

        // Handle form submission
        searchForm.addEventListener('submit', async function(e) {
            e.preventDefault();

            const awcCode = anganwadiSelect.value;
            if (!awcCode) return;

            // Show loading
            showAlert('Loading data...', 'info');
            searchBtn.disabled = true;

            try {
                const formData = new FormData();
                formData.append('awc_code', awcCode);

                const response = await fetch('<?= base_url('home/get-children-data') ?>', {
                    method: 'POST',
                    body: formData
                });

                const data = await response.json();

                // Update stats
                document.getElementById('totalChildren').textContent = data.children_data.length;
                document.getElementById('totalOutreach').textContent = data.outreach_data.length;
                document.getElementById('statsSection').classList.add('show');

                // Display outreach data
                displayOutreachData(data.outreach_data);

                // Display children data
                displayChildrenData(data.children_data);

                hideAlert();
                searchBtn.disabled = false;

            } catch (error) {
                console.error('Error:', error);
                showAlert('Error loading data', 'warning');
                searchBtn.disabled = false;
            }
        });

        function displayOutreachData(data) {
            const tbody = document.getElementById('outreachTableBody');
            tbody.innerHTML = '';

            if (data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" class="no-data">No outreach data found</td></tr>';
            } else {
                data.forEach(item => {
                    const row = `
                        <tr>
                            <td>${item.id || '-'}</td>
                            <td>${item.anganwadi_center_code || '-'}</td>
                            <td>${item.awc_worker_name || '-'}</td>
                            <td>${item.address_of_new_locations || '-'}</td>
                            <td>${item.building_type || '-'}</td>
                            <td>${item.location_area || '-'}</td>
                            <td>${item.month_year || '-'}</td>
                            <td>${item.created_at || '-'}</td>
                        </tr>
                    `;
                    tbody.innerHTML += row;
                });
            }

            document.getElementById('outreachSection').classList.add('show');
        }

        function displayChildrenData(data) {
            const tbody = document.getElementById('childrenTableBody');
            tbody.innerHTML = '';

            if (data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="12" class="no-data">No children data found</td></tr>';
            } else {
                data.forEach(child => {
                    const row = `
                        <tr>
                            <td>${child.child_Id || '-'}</td>
                            <td>${child.name || '-'}</td>
                            <td>${child.gender || '-'}</td>
                            <td>${child.dob || '-'}</td>
                            <td>${child.age_in_months || '-'}</td>
                            <td>${child.father_name || '-'}</td>
                            <td>${child.mother_name || '-'}</td>
                            <td>${child.mobile_no || '-'}</td>
                            <td>${child.awc_name || '-'}</td>
                            <td>${child.birth_weight || '-'}</td>
                            <td>${child.current_weight || '-'}</td>
                            <td>${child.category || '-'}</td>
                        </tr>
                    `;
                    tbody.innerHTML += row;
                });
            }

            document.getElementById('childrenSection').classList.add('show');
        }

        function showAlert(message, type) {
            const alertBox = document.getElementById('alertBox');
            alertBox.textContent = message;
            alertBox.className = 'alert alert-' + type + ' show';
        }

        function hideAlert() {
            const alertBox = document.getElementById('alertBox');
            alertBox.classList.remove('show');
        }
    </script>
</body>
</html>
