<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Import Monthly Outreach Data</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .container {
            background: white;
            border-radius: 10px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            max-width: 800px;
            width: 100%;
            padding: 40px;
        }

        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }

        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }

        .upload-section {
            border: 2px dashed #ddd;
            border-radius: 8px;
            padding: 40px;
            text-align: center;
            margin-bottom: 20px;
            transition: all 0.3s ease;
        }

        .upload-section:hover {
            border-color: #667eea;
            background: #f8f9ff;
        }

        .upload-section.dragover {
            border-color: #667eea;
            background: #f0f2ff;
        }

        .upload-icon {
            font-size: 48px;
            color: #667eea;
            margin-bottom: 20px;
        }

        .file-input-wrapper {
            position: relative;
            display: inline-block;
            cursor: pointer;
        }

        .file-input-wrapper input[type=file] {
            position: absolute;
            left: -9999px;
        }

        .file-input-label {
            display: inline-block;
            padding: 12px 30px;
            background: #667eea;
            color: white;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .file-input-label:hover {
            background: #5568d3;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }

        .file-name {
            display: block;
            margin-top: 15px;
            color: #666;
            font-size: 14px;
        }

        .btn {
            padding: 12px 30px;
            border: none;
            border-radius: 6px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-block;
        }

        .btn-primary {
            background: #667eea;
            color: white;
        }

        .btn-primary:hover:not(:disabled) {
            background: #5568d3;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }

        .btn-secondary {
            background: #e0e0e0;
            color: #333;
            margin-left: 10px;
        }

        .btn-secondary:hover {
            background: #d0d0d0;
        }

        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .button-group {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 6px;
            margin-bottom: 20px;
            display: none;
        }

        .alert.show {
            display: block;
        }

        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .alert-info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }

        .progress-bar {
            width: 100%;
            height: 6px;
            background: #e0e0e0;
            border-radius: 3px;
            overflow: hidden;
            margin-top: 20px;
            display: none;
        }

        .progress-bar.show {
            display: block;
        }

        .progress-bar-fill {
            height: 100%;
            background: #667eea;
            transition: width 0.3s ease;
            width: 0%;
        }

        .results {
            margin-top: 30px;
            display: none;
        }

        .results.show {
            display: block;
        }

        .result-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }

        .stat-card {
            padding: 20px;
            border-radius: 6px;
            text-align: center;
        }

        .stat-card.success {
            background: #d4edda;
            color: #155724;
        }

        .stat-card.error {
            background: #f8d7da;
            color: #721c24;
        }

        .stat-number {
            font-size: 32px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            opacity: 0.8;
        }

        .error-list {
            max-height: 300px;
            overflow-y: auto;
            background: #f8f9fa;
            padding: 15px;
            border-radius: 6px;
            font-size: 14px;
        }

        .error-list ul {
            list-style-position: inside;
            color: #721c24;
        }

        .error-list li {
            margin-bottom: 8px;
        }

        .template-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 6px;
            margin-top: 30px;
        }

        .template-section h3 {
            color: #333;
            margin-bottom: 10px;
            font-size: 18px;
        }

        .template-section p {
            color: #666;
            margin-bottom: 15px;
            font-size: 14px;
        }

        @media (max-width: 600px) {
            .container {
                padding: 20px;
            }

            h1 {
                font-size: 24px;
            }

            .button-group {
                flex-direction: column;
            }

            .btn-secondary {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Import Monthly Outreach Data</h1>
        <p class="subtitle">Upload a CSV file to import monthly outreach data into the system</p>

        <div id="alertBox" class="alert"></div>

        <div class="upload-section" id="uploadSection">
            <div class="upload-icon">📁</div>
            <div class="file-input-wrapper">
                <input type="file" id="csvFile" accept=".csv" onchange="handleFileSelect(event)">
                <label for="csvFile" class="file-input-label">Choose CSV File</label>
            </div>
            <span class="file-name" id="fileName">No file chosen</span>
        </div>

        <div class="progress-bar" id="progressBar">
            <div class="progress-bar-fill" id="progressBarFill"></div>
        </div>

        <div class="button-group">
            <button class="btn btn-primary" id="uploadBtn" onclick="uploadFile()" disabled>
                Upload & Import
            </button>
            <button class="btn btn-secondary" onclick="resetForm()">
                Reset
            </button>
        </div>

        <div class="results" id="results">
            <h3>Import Results</h3>
            <div class="result-stats">
                <div class="stat-card success">
                    <div class="stat-number" id="importedCount">0</div>
                    <div class="stat-label">Records Imported</div>
                </div>
                <div class="stat-card error">
                    <div class="stat-number" id="failedCount">0</div>
                    <div class="stat-label">Records Failed</div>
                </div>
            </div>
            <div id="errorSection" class="error-list" style="display: none;">
                <h4>Errors:</h4>
                <ul id="errorList"></ul>
            </div>
        </div>

        <div class="template-section">
            <h3>Need a Template?</h3>
            <p>Download a CSV template with the correct column headers and sample data</p>
            <a href="<?= base_url('import/download-template') ?>" class="btn btn-secondary" download>
                Download CSV Template
            </a>
        </div>
    </div>

    <script>
        let selectedFile = null;

        // Handle file selection
        function handleFileSelect(event) {
            selectedFile = event.target.files[0];
            if (selectedFile) {
                document.getElementById('fileName').textContent = selectedFile.name;
                document.getElementById('uploadBtn').disabled = false;
            } else {
                document.getElementById('fileName').textContent = 'No file chosen';
                document.getElementById('uploadBtn').disabled = true;
            }
        }

        // Handle drag and drop
        const uploadSection = document.getElementById('uploadSection');

        uploadSection.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadSection.classList.add('dragover');
        });

        uploadSection.addEventListener('dragleave', () => {
            uploadSection.classList.remove('dragover');
        });

        uploadSection.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadSection.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files.length > 0 && files[0].name.endsWith('.csv')) {
                document.getElementById('csvFile').files = files;
                handleFileSelect({ target: { files: files } });
            } else {
                showAlert('Please drop a valid CSV file', 'error');
            }
        });

        // Upload file
        async function uploadFile() {
            if (!selectedFile) {
                showAlert('Please select a file first', 'error');
                return;
            }

            const formData = new FormData();
            formData.append('csv_file', selectedFile);

            // Show progress bar
            document.getElementById('progressBar').classList.add('show');
            document.getElementById('progressBarFill').style.width = '50%';
            document.getElementById('uploadBtn').disabled = true;

            try {
                const response = await fetch('<?= base_url('import/csv') ?>', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();

                // Complete progress
                document.getElementById('progressBarFill').style.width = '100%';

                // Hide progress bar after a delay
                setTimeout(() => {
                    document.getElementById('progressBar').classList.remove('show');
                    document.getElementById('progressBarFill').style.width = '0%';
                }, 1000);

                if (result.success) {
                    showAlert(result.message, 'success');
                    displayResults(result);
                } else {
                    showAlert(result.message, 'error');
                    if (result.errors && Object.keys(result.errors).length > 0) {
                        let errorMsg = '';
                        for (let field in result.errors) {
                            errorMsg += result.errors[field] + '<br>';
                        }
                        document.getElementById('alertBox').innerHTML += '<br>' + errorMsg;
                    }
                }

            } catch (error) {
                document.getElementById('progressBar').classList.remove('show');
                showAlert('Error uploading file: ' + error.message, 'error');
            }

            document.getElementById('uploadBtn').disabled = false;
        }

        // Display results
        function displayResults(result) {
            document.getElementById('importedCount').textContent = result.imported;
            document.getElementById('failedCount').textContent = result.failed;
            document.getElementById('results').classList.add('show');

            if (result.errors && result.errors.length > 0) {
                const errorList = document.getElementById('errorList');
                errorList.innerHTML = '';
                result.errors.forEach(error => {
                    const li = document.createElement('li');
                    li.textContent = error;
                    errorList.appendChild(li);
                });
                document.getElementById('errorSection').style.display = 'block';
            } else {
                document.getElementById('errorSection').style.display = 'none';
            }
        }

        // Show alert
        function showAlert(message, type) {
            const alertBox = document.getElementById('alertBox');
            alertBox.textContent = message;
            alertBox.className = 'alert alert-' + type + ' show';
            
            setTimeout(() => {
                alertBox.classList.remove('show');
            }, 5000);
        }

        // Reset form
        function resetForm() {
            document.getElementById('csvFile').value = '';
            document.getElementById('fileName').textContent = 'No file chosen';
            document.getElementById('uploadBtn').disabled = true;
            document.getElementById('results').classList.remove('show');
            document.getElementById('alertBox').classList.remove('show');
            selectedFile = null;
        }
    </script>
</body>
</html>
